import React, {Component} from 'react';
import PropTypes from 'prop-types';
import {FormField} from 'react-form';
import {withApollo} from "react-apollo";
import {AsyncTypeahead} from "react-bootstrap-typeahead";
import classNames from 'classnames';
import {FormText} from "reactstrap";

const DEFAULT_LIMIT = 20;

class TypeaheadWrapper extends Component {
    constructor(props) {
        super(props);
        this.state = {
            loading: false,
            options: [],
            offset: 0,
            value: "",
        };

        this.onSearch = this.onSearch.bind(this);
        this.fetchData = this.fetchData.bind(this);
        this.onPaginate = this.onPaginate.bind(this);
        this.onInputChange = this.onInputChange.bind(this);
    }

    fetchData(page = 0) {
        const {client, dataQuery} = this.props;
        const {value} = this.state;

        if(!dataQuery) {
            throw new Error("dataQuery prop is required");
        }

        if(value.length < 3) {
            return;
        }

        this.setState({loading: true});
        let variables = {
            /*pagination: {
                limit: DEFAULT_LIMIT,
                offset: page,
            },*/
        };

        variables[this.props.filterKey] = value;

        client.query({
            query: dataQuery,
            variables: variables
        }).then(data => {
            this.setState({
                loading: false,
                options: data.data[this.props.resultKey],
            });
        });
    }

    onInputChange(value) {
        this.setState({value: value});
    }

    onPaginate(event) {
        const {offset} = this.state;
        this.fetchData(offset + 1);
    }

    onSearch(value) {
        this.fetchData(0);
    }

    render() {
        const {
            loading,
            options
        } = this.state;


        const {
            fieldApi,
            onInput,
            className,
            helpText,
            ...rest
        } = this.props;

        const {
            getValue,
            getError,
            getWarning,
            getTouched,
            getSuccess,
            setValue,
            setTouched,
        } = fieldApi;

        const touched = getTouched();
        const error = getError();
        const warning = getWarning();
        const success = getSuccess();

        return (
            <div>
                <AsyncTypeahead
                    onSearch={this.onSearch}
                    onInputChange={this.onInputChange}
                    onChange={(values) => {
                        setValue(values);

                        if ( onInput ) {
                            onInput( values );
                        }
                    }}
                    onBlur={() => {
                        setTouched();
                    }}
                    useCache={false}
                    options={options}
                    isLoading={loading}
                    selected={getValue() || []}
                    className={classNames(className, {'is-invalid': error && touched})}
                    {...rest} />
                { error ? <div className="invalid-feedback">{error}</div> : null }
                { helpText ? <FormText>{helpText}</FormText> : null}
            </div>
        );
    }
}

const Typeahead = FormField(withApollo(TypeaheadWrapper));

Typeahead.defaultProps = {};

Typeahead.propTypes = {
    fieldApi: PropTypes.object,
    helpText: PropTypes.string,
    dataQuery: PropTypes.object.isRequired,
    resultKey: PropTypes.string.isRequired,
    filterKey: PropTypes.string.isRequired,

};

export default Typeahead;